/*
 * Copyright (C) Jan 2006 Mellanox Technologies Ltd. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 *  utils.cpp - Miscellaneous general routines
 *
 *  Version: $Id: utils.cpp 2752 2006-01-19 14:40:17Z mst $
 *
 */

#define __EXTENSIONS__
#include <stdio.h>
#include <stdarg.h>

#include "utils.h"
#include "compatibility.h"

namespace std {}; using namespace std;

//--------------------------------------------------------
//------------------------------ error reporting utilities
//--------------------------------------------------------
//
////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////
// don't forget to delete char* ptr returned from this!!!
char *vprint(const char *format, va_list args)
{
    const int INIT_VAL = 1024;
    int       max_str, max_buf = INIT_VAL;
    char      *out_buf;

    while (1)
    {
        out_buf = new char[max_buf];
        max_str = max_buf - 1;

        if (vsnprintf(out_buf, max_str, format, args) < max_str)
            return out_buf;
        delete [] out_buf;
        max_buf *= 2;
    }
} // vprint


////////////////////////////////////////////////////////////////////////
// don't forget to delete char* ptr returned from this!!!
char *print(const char *format, ...)
{
    va_list   args;

    va_start(args, format);
    char *rc = vprint(format, args);
    va_end(args);

    return rc;
} // print


//-------------------------------------------------------
//------------------------------ string parsing utilities
//-------------------------------------------------------
//
////////////////////////////////////////////////////////////////////////
string next_token(string& str, const char *delimiters)
{
    if (str.empty())
        return "";

    // Special case - delimiters are empty. The token is next character
    if (!*delimiters)
    {
        char c = str[0];
        str.erase(0, 1);
        return string(&c, 1);
    }

    string::size_type beg = str.find_first_not_of(delimiters);
    if (beg == string::npos)
        return "";
    string::size_type end = str.find_first_of(delimiters, beg);
    if (end == string::npos)
    {
        string rc = str.substr(beg);
        str = "";
        return rc;
    }
    string rc = str.substr(beg, end-beg);
    str.erase(0, end);
    return rc;
} // next_token


////////////////////////////////////////////////////////////////////////
template <class Cont>
Cont split(const string& s, const char *delimiters, bool allow_empty)
{
    Cont lst;

    // Special case - delimiters are empty. Split to individual characters
    if (!*delimiters)
    {
        for (string::size_type k=0; k < s.length(); ++k)
        {
            char c = s[k];
            lst.push_back(string(&c, 1));
        }
        return lst;
    }

    // Begining of field
    string::size_type j = s.find_first_not_of(delimiters);

    // String consists only from delimiters
    if (j == string::npos)
    {
        if (allow_empty)
            for (string::size_type k=0; k < s.length(); ++k)
                lst.push_back("");
        return lst;
    } else if (allow_empty  &&  j > 0)
        for (string::size_type k=0; k < j; ++k)
            lst.push_back("");

    // Begining of delimiters
    string::size_type i = s.find_first_of(delimiters, j);

    while (i != string::npos)
    {
        lst.push_back(s.substr(j, i - j));
        j = s.find_first_not_of(delimiters, i);

        // Rest of string - delimiters only
        if (j == string::npos)
        {
            if (allow_empty)
                for (string::size_type k=i; k < s.length(); ++k)
                    lst.push_back("");
            return lst;
        }
        else if (allow_empty  &&  j > i+1)
            for (string::size_type k=i+1; k < j; ++k)
                lst.push_back("");

        // Next begining of delimiters
        i = s.find_first_of(delimiters, j);
    }

    lst.push_back(s.substr(j));
    return lst;
} // split


////////////////////////////////////////////////////////////////////////
list<string> splitl(const string& s, const char *delimiters, bool allow_empty)
{
    return split<list<string> >(s, delimiters, allow_empty);
} // split_v


////////////////////////////////////////////////////////////////////////
vector<string> splitv(const string& s, const char *delimiters, bool allow_empty)
{
    return split<vector<string> >(s, delimiters, allow_empty);
} // split_l


#ifndef NO_REGEX

//-------------------------------------------
//------------------------------ Regexp class
//-------------------------------------------
//
///////////////////////////////////////////////////////////////////////
bool Regexp::compile(const char *str_re, const int flags)
{
    int rc = regcomp(&_re, str_re, flags | REG_EXTENDED);
    _compiled = true;

    if (rc)
    {
        const unsigned INIT_VAL = 1024;
        unsigned       max_str, max_buf = INIT_VAL;
        char           *out_buf;

        while (1)
        {
            out_buf = new char[max_buf];
            max_str = max_buf - 1;

            if (regerror(rc, &_re, out_buf, max_str) < max_str)
            {
                _err =  out_buf;
                break;
            }
            delete [] out_buf;
            max_buf *= 2;
        }
        _valid = false;
    }
    else
        _valid = true;

    return _valid;
} // Regexp::Regex


////////////////////////////////////////////////////////////////////////
Regexp::~Regexp()
{
    if (_compiled)
    {
        regfree(&_re);
        _compiled = false;
    }
    err_clear();
} // Regexp::~Regexp

////////////////////////////////////////////////////////////////////////
bool Regexp::match(const std::string& s)
{
    // Regex is not valid (or never been compiled)
    if (!_compiled  ||  !_valid)
        return false;

    _matches.clear();
    vector<regmatch_t> match(_re.re_nsub + 1);
    if (regexec(&_re, s.c_str(), _re.re_nsub + 1, &match[0], 0))
        return false;

    for (unsigned i = 0; i<_re.re_nsub + 1; i++)
    {
        if (match[i].rm_so == -1)
            break;
        _matches.push_back(s.substr(match[i].rm_so,
                                    match[i].rm_eo - match[i].rm_so));
    }
    return true;
} // Regexp::match


////////////////////////////////////////////////////////////////////////
string Regexp::cap(const unsigned n)
{
    // Regex is not valid (or never been compiled)
    if (!_compiled  ||  !_valid)
        return "";
    if (n >= _matches.size())
        return "";
    return _matches[n];
} // Regexp::cap


////////////////////////////////////////////////////////////////////////
template <class Cont> Cont Regexp::split(const string& s)
{
    Cont         lst;
    regmatch_t   match[1];

    // Special case - regex is not valid (or never been compiled)
    if (!_compiled  ||  !_valid)
    {
        lst.push_back(s);
        return lst;
    }

    string str = s;
    while (!regexec(&_re, str.c_str(), 1, match, 0))
    {
        if (match[0].rm_so != -1  &&  match[0].rm_so > 0)
            lst.push_back(str.substr(0, match[0].rm_so));
        str = str.substr(match[0].rm_eo);
        if (str.empty())
            return lst;
    }

    lst.push_back(str);
    return lst;
} // Regexp::split


////////////////////////////////////////////////////////////////////////
list<string> Regexp::splitl(const string& s)
{
    return split<list<string> >(s);
} // Regexp::splitl


////////////////////////////////////////////////////////////////////////
vector<string> Regexp::splitv(const string& s)
{
    return split<vector<string> >(s);
} // Regexp::splitv

#endif //NO_REGEX


//-------------------------------------------
//------------------------------ ErrMsg class
//-------------------------------------------
//
////////////////////////////////////////////////////////////////////////
bool ErrMsg::errmsg(const char *format, ...)
{
    va_list   args;

    va_start(args, format);
    _err = vprint(format, args);
    va_end(args);

    //printf("-D- Error set for %p: %s\n", this, _err); // !!! Need to check error handling hierarchy 
    return false;
} // Special::errmsg



///////////////////////////////////////
//
// Endianess issues - Global << operators.
//
///////////////////////////////////////


vector<u_int8_t>&  operator<<(vector<u_int8_t>& v1, u_int32_t w)
{
    u_int32_t val = __cpu_to_be32(w);
    u_int8_t  *pw = (u_int8_t*)&val;
    v1.insert(v1.end(), pw, pw+sizeof(u_int32_t));
    return v1;
}
        
vector<u_int8_t>&  operator<<(vector<u_int8_t>& v1, u_int8_t b)
{
    v1.push_back(b);
    return v1;
}

vector<u_int8_t>&  operator<<(vector<u_int8_t>& v1, vector<u_int8_t>& v2)
{
    v1.insert(v1.end(), v2.begin() , v2.end() );
    return v1;
}

vector<u_int8_t>&  operator<<(vector<u_int8_t>& v1, vector<u_int32_t>& v2)
{
    for (u_int32_t i= 0 ; i < v2.size() ; i++)
        v1 << v2[i];
    return v1;
}


